/* wmle_ability.c
 *
 * Copyright (C) 2005, 2006, 2007 Stephane Germain
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/**
   \file
   \brief Warm's Weighted Maximum Likelihood of the abilities.
   \author Stephane Germain <germste@gmail.com>
*/

#include "libirt.h"
#if HAVE_CONFIG_H
#  include <config.h>
#endif
#include <math.h>
#include <gsl/gsl_spline.h>
#include <gsl/gsl_errno.h>
#include <gsl/gsl_roots.h>

/**
   \brief Used to passed extra parameter to \em wmle_ability_fdfdf2.

   This is used to comply with the root finding functions in
   the gsl (GNU scientific library).
*/
typedef struct
{
  /** \brief The spline interpolation of the test information. */
  gsl_spline *I_spline;

  /** \brief The spline interpolation of the likelihood. */
  gsl_spline *L_spline;

  /** \brief The accelerator of the spline interpolation. */
  gsl_interp_accel *acc;

} wmle_ability_struct;

/**
   \brief Compute the log weighted likelihood, gradient and Hessian of the pattern.

   @param[in] ability The ability level.
   @param[in] params The extra parameter to passes to the function.
   @param[out] f The log weighted likelihood.
   @param[out] df The gradient of the log weighted likelihood.
   @param[out] df2 The Hessian of the log weighted likelihood.

   This function is not used directly by the root finding functions,
   but by others functions that comply with the gsl.
*/
void
wmle_ability_fdfdf2 (double ability, void *params,
			 double *f, double *df, double *df2)
{
  gsl_spline *I_spline = ((wmle_ability_struct *) params)->I_spline;
  gsl_spline *L_spline = ((wmle_ability_struct *) params)->L_spline;
  gsl_interp_accel *acc = ((wmle_ability_struct *) params)->acc;
  double I, I1, I11, L, L1, L11;

  I = gsl_spline_eval(I_spline, ability, acc);;
  I1 = gsl_spline_eval_deriv(I_spline, ability, acc);;
  I11 = gsl_spline_eval_deriv2(I_spline, ability, acc);;
  L = gsl_spline_eval(L_spline, ability, acc);;
  L1 = gsl_spline_eval_deriv(L_spline, ability, acc);;
  L11 = gsl_spline_eval_deriv2(L_spline, ability, acc);;

  if (f)
    *f = log(I)/2 + log(L);
  if (df)
    *df = I1/(2*I) + L1/L;
  if (df2)
    *df2 = I11/(2*I)-I1*I1/(2*I*I) + L11/L-L1*L1/(L*L);
}

/**
   \brief Compute the log weighted likelihood of the pattern.

   @param[in] ability The ability level.
   @param[in] params The extra parameter to passes to the function.

   This function is just a wrapper around \em wmle_ability_fdfdf2
   to be used by the root finding functions in the gsl.

   \return The log weighted likelihood.
*/
double
wmle_ability_f (double ability, void *params)
{
  double f;
  wmle_ability_fdfdf2 (ability, params, &f, NULL, NULL);
  return f;
}

/**
   \brief Compute the log weighted likelihood gradient of the pattern.

   @param[in] ability The ability level.
   @param[in] params The extra parameter to passes to the function.

   This function is just a wrapper around \em wmle_ability_fdfdf2
   to be used by the root finding functions in the gsl.

   \return The log weighted likelihood gradient.
*/
double
wmle_ability_df (double ability, void *params)
{
  double df;
  wmle_ability_fdfdf2 (ability, params, NULL, &df, NULL);
  return df;
}

/**
   \brief Compute the log weighted likelihood Hessian of the pattern.

   @param[in] ability The ability level.
   @param[in] params The extra parameter to passes to the function.

   This function is just a wrapper around \em wmle_ability_fdfdf2
   to be used by the root finding functions in the gsl.

   \return The log weighted likelihood Hessian.
*/
double
wmle_ability_df2 (double ability, void *params)
{
  double df2;
  wmle_ability_fdfdf2 (ability, params, NULL, NULL, &df2);
  return df2;
}

/**
   \brief Compute the log weighted likelihood gradient and Hessian of the pattern.

   @param[in] ability The ability level.
   @param[in] params The extra parameter to passes to the function.
   @param[out] df The gradient of the log weighted likelihood.
   @param[out] df2 The Hessian of the log weighted likelihood.

   This function is just a wrapper around \em wmle_ability_fdfdf2
   to be used by the root finding functions in the gsl.
*/
void
wmle_ability_dfdf2 (double ability, void *params, double *df, double *df2)
{
  wmle_ability_fdfdf2 (ability, params, NULL, df, df2);
}

/**
   \brief Compute the WMLE of the abilities.

   @param[in] max_iter The maximum number of Newton iterations performed for each pattern.
   @param[in] prec The desired precision of each parameter estimate.
   @param[in] like A matrix (patterns x classes) of likelihood functions.
   @param[in] info A vecor (classes) with the test information function.
   @param[in] quad_points A vector(classes) with the middle points of each class.
   @param[out] abilities A vector(patterns) with the estimated abilities.
   @param[out] abilities_stddev A vector(items) with the standard errors
   of the estimated abilities.

   \warning The memory for \em abilities and \em abilities_stddev should be allocated before.
*/
int
wmle_abilities (int max_iter, double prec,
		gsl_matrix * like, gsl_vector * info, gsl_vector * quad_points,
		gsl_vector * abilities, gsl_vector * abilities_stddev)
{
  int nbr_quad = quad_points->size, j, nbr_pattern = like->size1,
    status, iter, ret_val;
  double ability, ability0;
  gsl_interp_accel *acc = gsl_interp_accel_alloc ();
  gsl_spline *I_spline = gsl_spline_alloc (gsl_interp_cspline, nbr_quad);
  gsl_spline *L_spline = gsl_spline_alloc (gsl_interp_cspline, nbr_quad);
  const gsl_root_fdfsolver_type *T;
  gsl_root_fdfsolver *s;
  gsl_function_fdf FDF;
  wmle_ability_struct params;

  /* set parameters and function to solve */
  params.I_spline = I_spline;
  params.L_spline = L_spline;
  params.acc = acc;
  FDF.fdf = &wmle_ability_dfdf2;
  FDF.f = &wmle_ability_df;
  FDF.df = &wmle_ability_df2;
  FDF.params = &params;

  /* select the algorithm to used */
  T = gsl_root_fdfsolver_steffenson;
  /* allocate the solver */
  s = gsl_root_fdfsolver_alloc (T);

  /* interpolate the test information */
  gsl_spline_init (I_spline, quad_points->data, info->data, nbr_quad);

  ret_val = 0;

  /* for each pattern */
  for (j = 0; j < nbr_pattern; j++)
    {
      if (libirt_verbose > 3)
	printf ("pattern %d ", j + 1);

      /* set the starting value */
      ability = gsl_vector_get (abilities, j);

      /* interpolate the likelihood */
      gsl_spline_init (L_spline, quad_points->data, like->data+j*nbr_quad, nbr_quad);

      /* set the solver */
      gsl_root_fdfsolver_set (s, &FDF, ability);

      /* iterate the solver */
      iter = 0;
      do
	{
	  iter++;
	  status = gsl_root_fdfsolver_iterate (s);
	  ability0 = ability;
	  ability = gsl_root_fdfsolver_root (s);

	  /* if the step is to big shrink it */
	  if (fabs (ability - ability0) > 0.5)
	    {
	      ability = ability0 + ((ability > ability0) ? 0.5 : -0.5);
	      gsl_root_fdfsolver_set (s, &FDF, ability);
	    }

	  /* check for out of bound */
	  if (ability < gsl_vector_get(quad_points, 1))
	    {
	      ability = gsl_vector_get(quad_points, 1);
	      gsl_root_fdfsolver_set (s, &FDF, ability);
	    }
	  if (ability > gsl_vector_get(quad_points, nbr_quad-2))
	    {
	      ability = gsl_vector_get(quad_points, nbr_quad-2);
	      gsl_root_fdfsolver_set (s, &FDF, ability);
	    }

	  /* test for convergence */
	  status = gsl_root_test_delta (ability, ability0, prec, 0);
	}
      while (status == GSL_CONTINUE && iter < max_iter);

      /* check if this pattern converged */
      if (status != GSL_SUCCESS)
	ret_val++;

      if (libirt_verbose > 3)
	{
	  if (status == GSL_CONTINUE)
	    {
	      printf ("did not converged (max iter)");
	    }
	  else if (status == GSL_EBADFUNC)
	    printf ("did not converged (singular point)");
	  else if (status == GSL_ENOPROG)
	    printf ("did not converged (no progress)");
	  else if (status == GSL_SUCCESS)
	    printf ("converged (success)");
	  else
	    printf ("unknow status");
	  printf (" after %d iterations.\n", iter);
	  fflush (stdout);
	}

      /* copy the solution found */
      gsl_vector_set (abilities, j, ability);

      /* copy the standard error */
      if (abilities_stddev)
	{
	  gsl_vector_set (abilities_stddev, j,
			  sqrt (1 / gsl_spline_eval(I_spline, ability, acc)));
	}
    }

  /* free the memory */
  gsl_root_fdfsolver_free (s);
  gsl_spline_free (I_spline);
  gsl_spline_free (L_spline);
  gsl_interp_accel_free (acc);

  return ret_val;
}

