/* mmle_2plm_mc.c
 *
 * Copyright (C) 2007 Stephane Germain
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/**
   \file
   \brief Functions to estimate the multiple choice item
   parameters by MMLE (Maximal Marginal Likelihood). The model is the
   multivariate logistic.

   The overall objectif is to find the OCC (option characteristic curves)
   maximizing the ML (marginal likelihood). An EM (expectation-maximization)
   iterative algorithm is used.

   1. A grid of ability levels has to be fixed. Something like 32 values from
   -4 to 4 will do. Those are called the quadrature classes in the code and
   can be generated with the function "quadrature".

   2. A first approximation of the OCC has to be available.

   3. For each pattern (a response vector from one subject) the a posteriori
   probabilities of being in each quadrature classes is computed by the
   function "posteriors".

   4. The expected number of subject in each quadrature classes
   (quad_sizes), and for each option the expected number of subject in
   each quadrature classes having choosen this option (quad_freqs) are
   computed by the function "frequencies".

   5. Once these quantities are assumed to be known, the log-likelihood can be
   maximized independantly for each item. The maximization is done by a root
   finding algorithm in the function "mle_2plm_mc". A variant of Newton-Raphson
   from the gsl library is used. For that we must have a function giving the
   firsts (gradient) and seconds (hessian) derivatives of the log-likelihood by
   the item parameters, those are computed in "like_2plm_mc_fdfdf2".

   6. Steps 3-5 are repeated until convergence is achieved.

   \author Stephane Germain <germste@gmail.com>
*/

#include "libirt.h"

#include <stdio.h>
#include <math.h>
#include <gsl/gsl_errno.h>
#include <gsl/gsl_multiroots.h>
#include <gsl/gsl_linalg.h>
#include <gsl/gsl_wavelet.h>

/**
   \brief Compute the response functions for a multivariate logistic model of one item.
   
   @param[in] slopes A vector(options) with the slope parameters of each option.
   @param[in] thresholds A vector(options) with the threshold parameters of each option.
   @param[in] quad_points A vector(classes) with the middle points of each quadrature class.
   @param[out] probs A matrix(options x classes) with the response functions.
   
   \todo Stddev of the probs

   \warning The memory for \em probs should be allocated before.
*/
void
prob_2plm_mc (gsl_vector * slopes, gsl_vector * thresholds,
	      gsl_vector * quad_points, gsl_matrix * probs)
{
  int nbr_quad, nbr_option, o, k;
  double prob, denom;

  nbr_quad = quad_points->size;
  nbr_option = slopes->size;

  /* for each class */
  for (k = 0; k < nbr_quad; k++)
    {
      /* compute the common denominator of the multivariate logistics */
      denom = 0.0;
      for (o = 0; o < nbr_option; o++)
	denom += exp(gsl_vector_get(slopes, o)
		     * (gsl_vector_get(quad_points, k)
			- gsl_vector_get(thresholds, o)));

      /* for each option */
      for (o = 0; o < nbr_option; o++)
	{
	  prob = exp(gsl_vector_get(slopes, o)
		     * (gsl_vector_get(quad_points, k)
			- gsl_vector_get(thresholds, o)))
	    / denom;
	  /* reset the prob if too small or too big */
	  if(prob > 1-VERY_SMALL_PROB) prob=1-VERY_SMALL_PROB;
	  if(prob < VERY_SMALL_PROB) prob=VERY_SMALL_PROB;
	  gsl_matrix_set (probs, o, k, prob);
	}
    }
}

/**
   \brief Compute the response functions for a multivariate logistic model on all the items.
   
   @param[in] slopes A vector(options) with the slope parameters of each option.
   @param[in] thresholds A vector(options) with the threshold parameters of each option.
   @param[in] nbr_options A vector(items) with the number of option of each items.
   @param[in] items_pos A vector(items) with the position of the first option of each item
   in patterns.
   @param[in] quad_points A vector(classes) with the middle points of each quadrature class.
   @param[out] probs A matrix(options x classes) with the response functions.
   
   \todo Stddev of the probs

   \warning The memory for \em probs should be allocated before.
*/
void
probs_2plm_mc (gsl_vector * slopes, gsl_vector * thresholds,
	       gsl_vector_int * nbr_options, gsl_vector_int * items_pos,
	       gsl_vector * quad_points, gsl_matrix * probs)
{
  int nbr_quad, nbr_item, nbr_option, pos, i;
  gsl_vector_view slopes_i, thresholds_i;
  gsl_matrix_view probs_i;

  nbr_quad = quad_points->size;
  nbr_item = nbr_options->size;

  /* for each item */
  for (i = 0; i < nbr_item; i++)
    {
      pos = gsl_vector_int_get(items_pos, i);
      nbr_option = gsl_vector_int_get(nbr_options, i);
      slopes_i = gsl_vector_subvector(slopes, pos, nbr_option);
      thresholds_i = gsl_vector_subvector(thresholds, pos, nbr_option);
      probs_i = gsl_matrix_submatrix(probs, pos, 0, nbr_option, nbr_quad);
      prob_2plm_mc(&slopes_i.vector, &thresholds_i.vector, quad_points, &probs_i.matrix);
    }
}

/**
   \brief Compute the gradient and Hessian of likelihood.

   @param[in] par The multivariate 2PLM parameters, first the (nbr_option-1)
   intercepts then the (nbr_option-1) slopes.
   @param[in] params The extra parameter to passes to the function.
   @param[out] df The gradient of the log likelihood.
   @param[out] df2 The Hessian of the log likelihood.

   This function is not used directly by the root finding functions,
   but by others functions that comply with the gsl.

   \return GSL_SUCCESS for success.
*/
int
like_2plm_mc_fdfdf2 (const gsl_vector * par, void *params,
		    double * f, gsl_vector * df, gsl_matrix * df2)
{
  gsl_vector *quad_points = ((like_2plm_mc_struct *) params)->quad_points;
  gsl_vector *quad_sizes = ((like_2plm_mc_struct *) params)->quad_sizes;
  gsl_matrix *quad_freqs = ((like_2plm_mc_struct *) params)->quad_freqs;
  double size, freq, grad, hess, theta, denom, prob, probj;
  int j, i, k, nbr_par = par->size, nbr_logit = nbr_par/2,
    nbr_quad = quad_freqs->size2;

  /* reset to zero */
  if (f) *f = 0;
  if (df)
    gsl_vector_set_all (df, 0.0);
  if (df2)
    gsl_matrix_set_all (df2, 0.0);

  /* sum over the classes */
  for (k = 0; k < nbr_quad; k++)
    {
      theta = gsl_vector_get (quad_points, k);      
      size = gsl_vector_get (quad_sizes, k);

      /* compute the common denominator of the multivariate logistics */
      denom = 1.0;
      for (i = 0; i < nbr_logit; i++)
	{
	  denom += exp(- gsl_vector_get(par, i)
		       - gsl_vector_get(par, nbr_logit+i) * theta);
	}

      /* for each logit (one less than the number of option) */
      for (i = 0; i < nbr_logit; i++)
	{
	  freq = gsl_matrix_get (quad_freqs, i, k);
	  prob = exp(- gsl_vector_get(par, i)
		     - gsl_vector_get(par, nbr_logit+i) * theta) / denom;
	  /* reset the prob if too small or too big */
	  if(prob > 1-VERY_SMALL_PROB) prob=1-VERY_SMALL_PROB;
	  if(prob < VERY_SMALL_PROB) prob=VERY_SMALL_PROB;
	  
	  /* update the llk */
	  if (f) {
	    *f += freq * log(prob);
	    if(gsl_isnan((*f))) return GSL_FAILURE;
	  }

	  /* update the gradient */
	  if (df)
	    {
	      /* intercept */
	      grad = gsl_vector_get (df, i);
	      grad -= (freq - size * prob);
	      gsl_vector_set (df, i, grad);
	      if(gsl_isnan(grad)) return GSL_FAILURE;

	      /* slope */
	      grad = gsl_vector_get (df, nbr_logit+i);
	      grad -= (freq - size * prob) * theta;
	      gsl_vector_set (df, nbr_logit+i, grad);
	      if(gsl_isnan(grad)) return GSL_FAILURE;
	    }
	  
	  /* update the Hessian */
	  if (df2)
	    {
	      /* for each logit j */
	      for (j = 0; j <= i; j++)
		{
		  probj = exp(- gsl_vector_get(par, j)
			      - gsl_vector_get(par, nbr_logit+j) * theta) / denom;
		  /* reset the prob if too small or too big */
		  if(probj > 1-VERY_SMALL_PROB) probj=1-VERY_SMALL_PROB;
		  if(probj < VERY_SMALL_PROB) probj=VERY_SMALL_PROB;

		  /* intercept(i) x intercept(j) */
		  hess = gsl_matrix_get (df2, i, j);
		  hess -= prob * ((i==j) - probj) * size;
		  gsl_matrix_set (df2, i, j, hess);
		  if(gsl_isnan(hess)) return GSL_FAILURE;

		  /* intercept(i) x slope(j) (transposed into the lower half) */
		  hess = gsl_matrix_get (df2, i, nbr_logit+j);
		  hess -= prob * ((i==j) - probj) * size * theta;
		  gsl_matrix_set (df2, nbr_logit+j, i, hess);
		  if(gsl_isnan(hess)) return GSL_FAILURE;

		  /* slope(i) x intercept(j) */
		  hess = gsl_matrix_get (df2, j, nbr_logit+i);
		  hess -= prob * ((i==j) - probj) * size * theta;
		  gsl_matrix_set (df2, nbr_logit+i, j, hess);
		  if(gsl_isnan(hess)) return GSL_FAILURE;

		  /* slope(i) x slope(j) */
		  hess = gsl_matrix_get (df2, nbr_logit+i, nbr_logit+j);
		  hess -= prob * ((i==j) - probj) * size * theta * theta;
		  gsl_matrix_set (df2, nbr_logit+i, nbr_logit+j, hess);
		  if(gsl_isnan(hess)) return GSL_FAILURE;
		}
	    }
	}

      /* last option */
      freq = gsl_matrix_get (quad_freqs, i, k);
      prob = 1.0 / denom;
      /* reset the prob if too small or too big */
      if(prob > 1-VERY_SMALL_PROB) prob=1-VERY_SMALL_PROB;
      if(prob < VERY_SMALL_PROB) prob=VERY_SMALL_PROB;
      /* update the llk for the last option */
      if (f){
	*f += freq * log(prob);
	if(gsl_isnan((*f))) return GSL_FAILURE;
      }

    }
  
  /* copy the lower half of the Hessian to the upper half */
  if (df2)
    for (i = 0; i < nbr_par; i++)
      for (j = 0; j < i; j++)
	gsl_matrix_set (df2, j, i, gsl_matrix_get (df2, i, j));

  return GSL_SUCCESS;
}

/**
   \brief Compute the gradient and Hessian of the likelihood.

   @param[in] par The parameters.
   @param[in] params The extra parameter to passes to the function.
   @param[out] df The gradient of the log likelihood.
   @param[out] df2 The Hessian of the log likelihood.

   This function is just a wrapper around \em like_2plmfdfdf2
   to be used by the root finding functions in the gsl.

   \return GSL_SUCCESS for success.
*/
int
like_2plm_mc_dfdf2 (const gsl_vector * par, void *params,
		    gsl_vector * df, gsl_matrix * df2)
{
  return like_2plm_mc_fdfdf2 (par, params, NULL, df, df2);
}

/**
   \brief Compute the gradient of the likelihood.

   @param[in] par The parameters.
   @param[in] params The extra parameter to passes to the function.
   @param[out] df The gradient of the log likelihood.

   This function is just a wrapper around \em like_2plmfdfdf2
   to be used by the root finding functions in the gsl.

   \return GSL_SUCCESS for success.
*/
int
like_2plm_mc_df (const gsl_vector * par, void *params, gsl_vector * df)
{
  return like_2plm_mc_fdfdf2 (par, params, NULL, df, NULL);
}

/**
   \brief Compute the Hessian of the likelihood.

   @param[in] par The parameters.
   @param[in] params The extra parameter to passes to the function.
   @param[out] df2 The Hessian of the log likelihood.

   This function is just a wrapper around \em like_2plmfdfdf2
   to be used by the root finding functions in the gsl.

   \return GSL_SUCCESS for success.
*/
int
like_2plm_mc_df2 (const gsl_vector * par, void *params, gsl_matrix * df2)
{
  return like_2plm_mc_fdfdf2 (par, params, NULL, NULL, df2);
}

/**
   \brief Does the maximization step of the EM algorithm to
   estimate the response functions by MMLE (Maximum Marginal Likelihood)
   of one multiple choice item.

   @param[in] max_iter The maximum number of Newton iterations performed for each item.
   @param[in] prec The desired precision of each parameter estimate.
   @param[in] params The extra parameter to passes to the function.
   @param[in,out] thresholds A vector(options) with the estimated thresholds.
   They should be initialize first.
   @param[out] thresh_stddev A vector(options) with the estimated thresholds standard deviation.
   @param[in,out] slopes A vector(options) with the estimated slopes.
   They should be initialize first.
   @param[out] slopes_stddev A vector(options) with the estimated slopes standard deviation.
   @param[out] mllk The maximum log likelihood.

   \return 1 if the item converge, 0 otherwise.
   
   \warning The memory for the outputs should be allocated before.
*/
int
mle_2plm_mc (int max_iter, double prec,
	     like_2plm_mc_struct * params,
	     gsl_vector * thresholds, gsl_vector * thresh_stddev,
	     gsl_vector * slopes, gsl_vector * slopes_stddev,
	     double * mllk)
{
  const gsl_multiroot_fdfsolver_type *algo;
  gsl_multiroot_fdfsolver *solver;
  int status, iter, nbr_option = thresholds->size, nbr_logit = nbr_option-1,
    i, j, k, ret_val, nbr_par = nbr_logit * 2;
  gsl_multiroot_function_fdf FDF;

  /* the vector with all the parameters */
  gsl_vector * par = gsl_vector_alloc(nbr_par);

  /* used to compute the standard errors */
  gsl_matrix *df2 = gsl_matrix_alloc (nbr_par, nbr_par);
  gsl_permutation *lu_perm = gsl_permutation_alloc (nbr_par);
  gsl_matrix *inv_df2 = gsl_matrix_alloc (nbr_par, nbr_par);
  gsl_matrix *G = gsl_matrix_alloc (nbr_par, nbr_option);
  int lu_sign;
  double var, sum_grad, mean_slope, mean_intercept;

  /* initalize the function to solve */
  FDF.f = &like_2plm_mc_df;
  FDF.df = &like_2plm_mc_df2;
  FDF.fdf = &like_2plm_mc_dfdf2;
  FDF.n = nbr_par;
  FDF.params = params;

  /* select the algorithm to used */
  algo = gsl_multiroot_fdfsolver_gnewton;

  /* allocate the solver */
  solver = gsl_multiroot_fdfsolver_alloc (algo, nbr_par);

  ret_val = 0;

  /* set the starting values */
  /* transform from centered thresholds and slopes to contrast intercepts and slopes */
  for (i = 0; i < nbr_logit; i++)
    {
      /* slopes */
      gsl_vector_set(par, nbr_logit+i,
		     gsl_vector_get(slopes, nbr_logit) - gsl_vector_get(slopes, i));

      /* intercepts */
      gsl_vector_set(par, i,
                     gsl_vector_get(slopes, i) * gsl_vector_get(thresholds, i) 
		     - gsl_vector_get(slopes, nbr_logit) * gsl_vector_get(thresholds, nbr_logit)); 
    }  

  /* set the solver */
  gsl_multiroot_fdfsolver_set (solver, &FDF, par);

  /* iterate the solver */
  iter = 0;
  do
    {
      iter++;
      
      /* check if the hessian is singular */
      status = 0;
      for (k = 0; k < nbr_par; k++)
	{
	  if(0 == gsl_matrix_get (solver->J, k, k))
	    {
	      status = GSL_EBADFUNC;
	      break;
	    }
	}
      if (status) break;

      status = gsl_multiroot_fdfsolver_iterate (solver);
      
      if (libirt_verbose > 9)
	{
	  sum_grad = 0;
	  for (k = 0; k < nbr_par; k++) sum_grad += fabs(gsl_vector_get(solver->f, k));
	  printf ("\n At N-R iteration %d sum|grad(ML)| is %8.2e.\n",
		  iter, sum_grad);
	}
      
      if (status)
	break;

      /* test for convergence */
      status = gsl_multiroot_test_delta (solver->dx, solver->x, prec, 0);
      /* status = gsl_multiroot_test_residual (solver->f, prec); */
      
    }
  while (status == GSL_CONTINUE && iter < max_iter);

  /* compute the maximum log likelihood to return */
  like_2plm_mc_fdfdf2 (solver->x, params, mllk, NULL, NULL);

  /* check if this item converged */
  if (status != GSL_SUCCESS)
    {
      ret_val = 1;
    }
  
  if (libirt_verbose > 3)
    {
      if (status == GSL_CONTINUE)
	printf (" did not converged (max iter)");
      else if (status == GSL_EBADFUNC)
	printf (" did not converged (singular point)");
      else if (status == GSL_ENOPROG)
	printf (" did not converged (no progress)");
      else if (status == GSL_ENOPROGJ)
	printf (" did not converged (jacobian no progress)");
      else if (status == GSL_SUCCESS)
	printf (" converged (success)");
      else
	printf (" unknow status (%d)", status);
      printf (" after %d iterations.\n", iter);
      fflush (stdout);
    }
  
  /* copy the solution found */
  /* transform from contrast intercepts and slopes to centered thresholds and slopes */
  mean_slope = 0;
  mean_intercept = 0;
  for (i = 0; i < nbr_logit; i++)
    {
      mean_slope += gsl_vector_get(solver->x, nbr_logit+i);
      mean_intercept += gsl_vector_get(solver->x, i);
    }
  mean_slope /= nbr_option;
  mean_intercept /= nbr_option;
  for (i = 0; i < nbr_logit; i++)
    {
      gsl_vector_set(slopes, i, mean_slope - gsl_vector_get(solver->x, nbr_logit+i));
      gsl_vector_set(thresholds, i, - (mean_intercept - gsl_vector_get(solver->x, i)) /
		     gsl_vector_get(slopes, i));
    }
  gsl_vector_set(slopes, nbr_logit, mean_slope);
  gsl_vector_set(thresholds, nbr_logit, - mean_intercept /
		 gsl_vector_get(slopes, nbr_logit));

  /* compute the standard errors */
  if (thresh_stddev && slopes_stddev)
    {
      /* get the Hessian */
      like_2plm_mc_df2 (solver->x, params, df2);

      /* inverse it */
      gsl_linalg_LU_decomp (df2, lu_perm, &lu_sign);
      gsl_linalg_LU_invert (df2, lu_perm, inv_df2);

      /* slopes */
      for (k = 0; k < nbr_option; k++) 
	{
	  var = 0;
	  for (i = 0; i < nbr_logit; i++)
	    for (j = 0; j < nbr_logit; j++)
	      var -= (1.0/nbr_option - (i==k))
		* (1.0/nbr_option - (j==k))
		* gsl_matrix_get(inv_df2, nbr_logit+i, nbr_logit+j);
	  gsl_vector_set(slopes_stddev, k, sqrt(var));
	}

      /* intercepts */

      /* compute the derivatives of the transformations
	 from the contrast intercepts and slopes to the centered thresholds
	 each transformation is in a column of G */
      for (k = 0; k < nbr_option; k++)
	for (i = 0; i < nbr_logit; i++)
	  {
	    /* derivatives by the intercepts */
	    gsl_matrix_set(G, i, k, (1.0/nbr_option-(k==i))
			   / gsl_vector_get(slopes, k));
	    /* derivatives by the slopes */
	    gsl_matrix_set(G, nbr_logit+i, k, -(1.0/nbr_option-(k==i))
			   * gsl_vector_get(thresholds, k)
			   / gsl_vector_get(slopes, k));
	  }
      
      /* compute the variance of each threshold by Taylor approximation */
      for (k = 0; k < nbr_option; k++)
	{
	  var = 0;
	  for (i = 0; i < nbr_par; i++)
	    for (j = 0; j < nbr_par; j++)
	      var -= gsl_matrix_get(G, i, k)
		* gsl_matrix_get(G, j, k)
		* gsl_matrix_get(inv_df2, i, j);
	  gsl_vector_set(thresh_stddev, k, sqrt(var));
	}
    }

  /* free the memory */
  gsl_multiroot_fdfsolver_free (solver);
  gsl_vector_free (par);
  gsl_matrix_free (df2);
  gsl_permutation_free (lu_perm);
  gsl_matrix_free (G);

  return ret_val;
}

/**
   \brief Estimate the options response functions by MMLE (Maximum Marginal Likelihood).

   @param[in] max_em_iter The maximum number of EM iterations. At least 20 iteration are made.
   @param[in] max_nr_iter The maximum number of Newton iterations performed
   for each item at each EM iteration.
   @param[in] prec The relative change in the likelihood to stop the EM algorithm.
   This value divided by 10 is also the desired precision of each parameter estimate.
   @param[in] patterns A matrix(patterns x options) of binary responses.
   @param[in] counts A vector(patterns) with the count of each pattern.
   If NULL the counts are assumed to be all 1.
   @param[in] quad_points A vector(classes) with the middle points of each quadrature class.
   @param[in] quad_weights A vector(classes) with the prior weights of each quadrature class.
   @param[in] items_pos A vector(items) with the position of the first option of each item
   in patterns (and probs).
   @param[in] nbr_options A vector(items) with the number of option of each item
   in patterns (and probs).
   @param[in,out] thresholds A vector(options) with the estimated thresholds.
   They should be initialize first.
   @param[out] thresh_stddev A vector(options) with the estimated thresholds standard deviation.
   @param[in,out] slopes A vector(options) with the estimated slopes.
   They should be initialize first.
   @param[out] slopes_stddev A vector(options) with the estimated slopes standard deviation.
   @param[in] ignore A vector(items) of ignore flag.
   @param[out] nbr_notconverge The number of items that didn't converged.
   @param[out] notconverge A vector(items) of flag set for the items that didn't converged.
   @param[in] adjust_weights Controls whether adjust the quadrature weights after each iteration.

   \return 1 if the relative change in the maximum log likelihood was less than prec
   else 0.
   
   \warning The memory for the outputs should be allocated before.
*/
int
mmle_2plm_mc (int max_em_iter, int max_nr_iter, double prec,
	      gsl_matrix_int * patterns, gsl_vector * counts,
	      gsl_vector * quad_points, gsl_vector * quad_weights, 
	      gsl_vector_int * items_pos, gsl_vector_int * nbr_options,
	      gsl_vector * thresholds, gsl_vector * thresh_stddev,
	      gsl_vector * slopes, gsl_vector * slopes_stddev,
	      gsl_vector_int * ignore,
	      int * nbr_notconverge, gsl_vector_int * notconverge,
	      int adjust_weights)
{
  int em_iter, nbr_quad, nbr_pattern, nbr_item, nbr_option_tot, nbr_option,
    ret_val, k, j, i, pos;
  double nbr_subject, mllk, mllk_old=0, mllk_i;
  gsl_matrix *quad_freqs, *post;
  gsl_vector *quad_sizes;
  gsl_matrix_view quad_freqs_i;
  gsl_vector_view thresh_i, thresh_stddev_i, slopes_i, slopes_stddev_i;
  like_2plm_mc_struct params;
  gsl_matrix *probs;

  nbr_quad = quad_points->size;
  nbr_pattern = patterns->size1;
  nbr_option_tot = patterns->size2;
  nbr_item = items_pos->size;

  nbr_subject = 0;
  /* count the number of subject */
  for(j = 0; j < nbr_pattern; j++)
    nbr_subject += counts ? gsl_vector_get(counts, j) : 1;

  /* allocate the memory */
  quad_freqs = gsl_matrix_alloc (nbr_option_tot, nbr_quad);
  quad_sizes = gsl_vector_alloc (nbr_quad);
  post = gsl_matrix_alloc (nbr_pattern, nbr_quad);
  probs = gsl_matrix_alloc (nbr_option_tot, nbr_quad);

  /* EM iterations */

  for (em_iter = 1; em_iter <= max_em_iter; em_iter++)
    {
      /* E (estimation) step */

      if (libirt_verbose > 2)
	printf ("\nEM iteration %d\n", em_iter);

      /* compute the response functions */
      probs_2plm_mc (slopes, thresholds, nbr_options, items_pos, quad_points, probs);

      /* compute the posterior prob */
      posteriors_mc (patterns, probs, nbr_options, items_pos, quad_weights, post);

      /* compute the expected sizes and frequencies */
      frequencies (patterns, counts, post, probs, quad_sizes, quad_freqs);

      /* print debugging information */
      if (libirt_verbose > 5)
	{
	  for (i = 0; i < nbr_item; i++)
	    for (j = 0; j < gsl_vector_int_get(nbr_options, i); j++)
	      {
		pos = gsl_vector_int_get(items_pos,i);
		printf("Probabilities for option %d of item %d :\n", j+1, i+1);
		for (k = 0; k < nbr_quad; k++)
		  printf(" %8.2e", gsl_matrix_get(probs, pos+j, k));
		printf("\n");
	      }
	  for (j = 0; j < nbr_pattern; j++)
	    {
	      printf("Posterior for pattern %d :\n", j+1);
	      for (k = 0; k < nbr_quad; k++)
		printf(" %8.2e", gsl_matrix_get(post,j,k));
	      printf("\n");
	    }
	  printf("Sizes :\n");
	  for (k = 0; k < nbr_quad; k++)
	    printf(" %8.2e", gsl_vector_get(quad_sizes,k));
	  printf("\n");
	  for (i = 0; i < nbr_item; i++)
	    for (j = 0; j < gsl_vector_int_get(nbr_options, i); j++)
	      {
		pos = gsl_vector_int_get(items_pos,i);
		printf("Frequencies for option %d of item %d :\n", j+1, i+1);
		for (k = 0; k < nbr_quad; k++)
		  printf(" %8.2e", gsl_matrix_get(quad_freqs,pos+j,k));
		printf("\n");
	      }
	}

      /* the number of item that do not converge */
      *nbr_notconverge = 0;

      mllk = 0;

      /* M (maximisation) step */
      for (i = 0; i < nbr_item; i++)
	{
	  /* ignore the degenerate items */
	  if (ignore && gsl_vector_int_get(ignore, i)) continue;

	  /* get the corresponding rows of freqs, sizes, probs and probs_stddev */
	  nbr_option = gsl_vector_int_get(nbr_options, i);
	  pos = gsl_vector_int_get(items_pos, i);
	  quad_freqs_i = gsl_matrix_submatrix(quad_freqs, pos, 0, nbr_option, nbr_quad);
	  thresh_i = gsl_vector_subvector(thresholds, pos, nbr_option);
	  thresh_stddev_i = gsl_vector_subvector(thresh_stddev, pos, nbr_option);
	  slopes_i = gsl_vector_subvector(slopes, pos, nbr_option);
	  slopes_stddev_i = gsl_vector_subvector(slopes_stddev, pos, nbr_option);
	  params.quad_freqs = &quad_freqs_i.matrix;
	  params.quad_sizes = quad_sizes;
	  params.quad_points = quad_points;

	  /* use a root finding algorithm */
	  if (libirt_verbose > 3)
	    printf ("item %d", i + 1);

	  ret_val = mle_2plm_mc
	    (max_nr_iter, prec/10, &params, &thresh_i.vector, &thresh_stddev_i.vector,
	     &slopes_i.vector, &slopes_stddev_i.vector, &mllk_i);
	  
	  *nbr_notconverge += ret_val;

	  mllk += mllk_i;
	  
	  gsl_vector_int_set(notconverge, i, ret_val);
	}

      if(gsl_isnan(mllk)) {
	if (libirt_verbose > 1) printf("NAN error ! Stopping.\n");
	break;
      }

      if(adjust_weights) 
	adjust_quad_weights (nbr_subject, quad_sizes, quad_points, quad_weights);

      if (libirt_verbose > 2)
	printf("MLLK = %10.3e %%CHANGE = %9.3e\n", mllk, fabs((mllk-mllk_old)/mllk));

      /* if the change in the maximum log likelihood is small then exit */
      if (fabs((mllk-mllk_old)/mllk) < prec && em_iter >= 20) break;

      mllk_old = mllk;
    }

  /* check if the EM algo converged */
  if (em_iter <= max_em_iter && !gsl_isnan(mllk)) ret_val = 1;
  else ret_val = 0;

  if (libirt_verbose > 0 && ret_val == 0)
    printf("The EM algorithm didn't converged after %d iterations.\n", em_iter-1);

  if (libirt_verbose > 0 && ret_val == 1)
    printf("The EM algorithm converged after %d iterations.\n", em_iter);

  /* free the memory */
  gsl_matrix_free (quad_freqs);
  gsl_vector_free (quad_sizes);
  gsl_matrix_free (post);
  gsl_matrix_free (probs);

  return ret_val;
}
